﻿//=============================================================================
// TMPlugin - マップＨＰゲージ (MZ版)
// バージョン: 2.0.0
// 最終更新日: 2024/11/18
// 配布元　　: https://hikimoki.sakura.ne.jp/
//-----------------------------------------------------------------------------
// Copyright (c) 2016 tomoaky
// Released under the MIT license.
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MZ
 * @plugindesc マップシーンに顔グラフィックとＨＰゲージを表示します。
 * @author tomoaky 改変：ユースフル小屋
 *
 * @param gaugeWindowX
 * @type number
 * @min -1000
 * @desc ＨＰゲージウィンドウのＸ座標
 * 初期値: 0
 * @default 0
 *
 * @param gaugeWindowY
 * @type number
 * @min -1000
 * @desc ＨＰゲージウィンドウのＹ座標
 * 初期値: 0
 * @default 0
 *
 * @param gaugeWindowWidth
 * @type number
 * @desc ＨＰゲージウィンドウの幅
 * 初期値: 288
 * @default 288
 *
 * @param gaugeWindowHeight
 * @type number
 * @desc ＨＰゲージウィンドウの高さ
 * 初期値: 64
 * @default 64
 * 
 * @param gaugeA
 * @type struct<Gauge>
 * @desc ゲージＡのパラメータ
 * @default {"type":"HP","x":"12","y":"12","width":"144","height":"36","fontSize":"28","param":"0","max":"0","name":"AP","color":"#ff60c0 #ffa0e0"}
 *
 * @param gaugeB
 * @type struct<Gauge>
 * @desc ゲージＢのパラメータ
 * @default {"type":"","x":"12","y":"12","width":"144","height":"36","fontSize":"28","param":"0","max":"0","name":"AP","color":"#ff60c0 #ffa0e0"}
 *
 * @param gaugeC
 * @type struct<Gauge>
 * @desc ゲージＣのパラメータ
 * @default {"type":"","x":"12","y":"12","width":"144","height":"36","fontSize":"28","param":"0","max":"0","name":"AP","color":"#ff60c0 #ffa0e0"}
 *
 * @param gaugeD
 * @type struct<Gauge>
 * @desc ゲージＤのパラメータ
 * @default {"type":"","x":"12","y":"12","width":"144","height":"36","fontSize":"28","param":"0","max":"0","name":"AP","color":"#ff60c0 #ffa0e0"}
 *
 * @param faceOffsetX
 * @type number
 * @min -1000
 * @desc 顔グラフィックのＸ座標補正値
 * 初期値: -4 ( -1000 で顔グラフィックを使用しない)
 * @default -4
 *
 * @param faceOffsetY
 * @type number
 * @min -1000
 * @desc 顔グラフィックのＹ座標補正値
 * 初期値: -4
 * @default -4
 *
 * @param stateIconMax
 * @type number
 * @desc ステートアイコンを表示する個数
 * 初期値: 4
 * @default 4
 *
 * @param stateIconX
 * @type number
 * @min -1000
 * @desc ステートアイコンのＸ座標
 * 初期値: 156
 * @default 156
 *
 * @param stateIconY
 * @type number
 * @min -1000
 * @desc ステートアイコンのＹ座標
 * 初期値: 24
 * @default 24
 * 
 * @param stateIconScale
 * @type number
 * @desc ステートアイコンの拡大率（ % ）
 * 初期値: 100
 * @default 100
 * 
 * @param stateIconOpacity
 * @type number
 * @desc ステートアイコンの不透明度（ 0 ~ 255 ）
 * 初期値: 255
 * @default 255
 *
 * @param goldWidth
 * @type number
 * @desc 所持金表示の幅
 * 初期値: 0 ( 0 で非表示 )
 * @default 0
 *
 * @param goldX
 * @type number
 * @min -9999
 * @desc 所持金表示のＸ座標
 * 初期値: 12
 * @default 12
 *
 * @param goldY
 * @type number
 * @min -1000
 * @desc 所持金表示のＹ座標
 * 初期値: 12
 * @default 12
 * 
 * @param vnMax
 * @type boolean
 * @desc ゲージタイプ VN の最大値を表示するかどうか
 * 初期値: OFF ( true = ON 表示 / false = OFF 非表示 )
 * @default false
 *
 * @param shakeTime
 * @type number
 * @desc ダメージを受けたときにウィンドウを揺らす時間（フレーム）
 * 初期値: 20 ( 0 で揺らさない )
 * @default 20
 *
 * @param startVisible
 * @type boolean
 * @desc ゲーム開始時の表示状態
 * 初期値: ON（ true = ON 表示 / false = OFF 非表示 ）
 * @default true
 *
 * @param windowOpacity
 * @type number
 * @desc ＨＰゲージウィンドウの不透明度
 * 初期値: 255
 * @default 255
 *
 * @param collideOpacity
 * @type number
 * @max 255
 * @desc プレイヤーと重なったときの不透明度
 * 初期値: 128（ 0 ～ 255 ）
 * @default 128
 *
 * @param messageBusyHide
 * @type boolean
 * @desc メッセージウィンドウ表示中はＨＰゲージウィンドウを隠す
 * 初期値: ON ( true = ON 隠す / false = OFF 隠さない )
 * @default true
 *
 * @param eventBusyHide
 * @type boolean
 * @desc イベント起動中はＨＰゲージウィンドウを隠す
 * 初期値: ON（ true = ON 隠す / false = OFF 隠さない )
 * @default true
 *
 * @param useBattleScene
 * @type boolean
 * @desc 戦闘シーンでもＨＰゲージウィンドウを表示する。
 * 初期値: OFF（ true = ON 表示 / false = OFF 非表示 )
 * @default false
 *
 * @param gaugeWindowBattleX
 * @type number
 * @min -1000
 * @desc 戦闘シーンのＨＰゲージウィンドウのＸ座標
 * 初期値: 0
 * @default 0
 *
 * @param gaugeWindowBattleY
 * @type number
 * @min -1000
 * @desc 戦闘シーンのＨＰゲージウィンドウのＹ座標
 * 初期値: 0
 * @default 0
 *
 * @command showHpGauge
 * @text HPゲージ表示
 * @desc HPゲージウィンドウを表示します
 * 
 * @arg gaugeId
 * @text ゲージID
 * @type select
 * @option すべて
 * @value 
 * @option A
 * @value A
 * @option B
 * @value B
 * @option C
 * @value C
 * @option D
 * @value D
 * @desc 表示するゲージ（空白ですべて表示）
 * @default 
 *
 * @command hideHpGauge
 * @text HPゲージ非表示
 * @desc HPゲージウィンドウを非表示にします
 * 
 * @arg gaugeId
 * @text ゲージID
 * @type select
 * @option すべて
 * @value 
 * @option A
 * @value A
 * @option B
 * @value B
 * @option C
 * @value C
 * @option D
 * @value D
 * @desc 非表示にするゲージ（空白ですべて非表示）
 * @default 
 *
 * @command moveHpGaugeWindow
 * @text HPゲージウィンドウ移動
 * @desc HPゲージウィンドウの位置を移動します
 * 
 * @arg x
 * @text X座標
 * @type number
 * @min -9999
 * @desc 移動先のX座標
 * @default 0
 * 
 * @arg y
 * @text Y座標
 * @type number
 * @min -9999
 * @desc 移動先のY座標
 * @default 0
 *
 * @help
 * TMPlugin - マップＨＰゲージ (MZ版) ver2.0.0
 * 
 */

/*~struct~Gauge:
 *
 * @param type
 * @type select
 * @option なし
 * @value 
 * @option HP
 * @option MP
 * @option TP
 * @option LV
 * @option VN
 * @desc ゲージのタイプ（HP / MP / TP / LV / VN）
 * 初期値: HP
 * @default HP
 *
 * @param x
 * @type number
 * @min -1000
 * @desc ゲージのＸ座標（ウィンドウ内の左端が 0 ）
 * 初期値: 12
 * @default 12
 *
 * @param y
 * @type number
 * @min -1000
 * @desc ゲージのＹ座標（ウィンドウ内の上端が 0 ）
 * 初期値: 12
 * @default 12
 *
 * @param width
 * @type number
 * @desc ゲージの長さ
 * 初期値: 144
 * @default 144
 *
 * @param height
 * @type number
 * @desc ゲージの表示領域（数値とゲージ合わせて）の高さ
 * 初期値: 36
 * @default 36
 *
 * @param fontSize
 * @type number
 * @desc フォントサイズ
 * 初期値: 28
 * @default 28
 *
 * @param param
 * @type variable
 * @desc ゲージのタイプが VN のときに現在値とするゲーム変数番号
 * 初期値: 0
 * @default 0
 *
 * @param max
 * @type variable
 * @desc ゲージのタイプが VN のときに最大値とするゲーム変数番号
 * 初期値: 0
 * @default 0
 *
 * @param name
 * @type string
 * @desc ゲージのタイプが VN のときに表示するパラメータ名
 * 初期値: AP
 * @default AP
 *
 * @param color
 * @type string
 * @desc ゲージのタイプが LV / VN のときのゲージカラー
 * 初期値: #ff60c0 #ffa0e0
 * @default #ff60c0 #ffa0e0
 */

var Imported = Imported || {};
Imported.TMMapHpGauge = true;

(() => {
	'use strict';

	const script = document.currentScript.src.split("/").pop().replace(/\.js$/, "");
	const parameters = PluginManager.parameters(script);
	
	const gaugeWindowX = Number(parameters['gaugeWindowX'] || 0);
	const gaugeWindowY = Number(parameters['gaugeWindowY'] || 0);
	const gaugeWindowWidth = Number(parameters['gaugeWindowWidth'] || 288);
	const gaugeWindowHeight = Number(parameters['gaugeWindowHeight'] || 64);
	
	const gauges = [];
	['A', 'B', 'C', 'D'].forEach((code, i) => {
		const gaugeParam = parameters['gauge' + code];
		if (gaugeParam) {
			gauges[i] = JSON.parse(gaugeParam);
			gauges[i].x = Number(gauges[i].x || 0);
			gauges[i].y = Number(gauges[i].y || 0);
			gauges[i].width = Number(gauges[i].width || 0);
			gauges[i].height = Number(gauges[i].height || 0);
			gauges[i].fontSize = Number(gauges[i].fontSize || 0);
			gauges[i].param = Number(gauges[i].param || 0);
			gauges[i].max = Number(gauges[i].max || 0);
			gauges[i].color = (gauges[i].color || '').split(' ');
		} else {
			gauges[i] = {type: '', x: 0, y: 0, width: 0, height: 0, fontSize: 0, param: 0, max: 0, name: '', color: []};
		}
	});
	
	const faceOffsetX = Number(parameters['faceOffsetX'] ?? -4);
	const faceOffsetY = Number(parameters['faceOffsetY'] ?? -4);
	const stateIconMax = Number(parameters['stateIconMax'] || 4);
	const stateIconX = Number(parameters['stateIconX'] || 156);
	const stateIconY = Number(parameters['stateIconY'] || 24);
	const stateIconScale = Number(parameters['stateIconScale'] || 100);
	const stateIconOpacity = Number(parameters['stateIconOpacity'] || 255);
	const goldWidth = Number(parameters['goldWidth'] || 0);
	const goldX = Number(parameters['goldX'] || 0);
	const goldY = Number(parameters['goldY'] || 0);
	const vnMax = parameters['vnMax'] === 'true';
	const shakeTime = Number(parameters['shakeTime'] || 20);
	const collideOpacity = Number(parameters['collideOpacity'] || 128);
	const startVisible = parameters['startVisible'] !== 'false';
	const windowOpacity = Number(parameters['windowOpacity'] || 255);
	const messageBusyHide = parameters['messageBusyHide'] !== 'false';
	const eventBusyHide = parameters['eventBusyHide'] !== 'false';
	const useBattleScene = parameters['useBattleScene'] === 'true';
	const gaugeWindowBattleX = Number(parameters['gaugeWindowBattleX'] || 0);
	const gaugeWindowBattleY = Number(parameters['gaugeWindowBattleY'] || 0);

	//-----------------------------------------------------------------------------
	// PluginManager
	//

	PluginManager.registerCommand(script, 'showHpGauge', args => {
		if (args.gaugeId) {
			const gaugeId = ['A', 'B', 'C', 'D'].indexOf(args.gaugeId);
			if (gaugeId >= 0) {
				$gameSystem.setVisibleMapHpGauges(gaugeId, true);
			}
		} else {
			$gameSystem.setVisibleMapHpGauge(true);
		}
	});

	PluginManager.registerCommand(script, 'hideHpGauge', args => {
		if (args.gaugeId) {
			const gaugeId = ['A', 'B', 'C', 'D'].indexOf(args.gaugeId);
			if (gaugeId >= 0) {
				$gameSystem.setVisibleMapHpGauges(gaugeId, false);
			}
		} else {
			$gameSystem.setVisibleMapHpGauge(false);
		}
	});

	PluginManager.registerCommand(script, 'moveHpGaugeWindow', args => {
		$gameSystem._mapHpGaugeWindowX = Number(args.x);
		$gameSystem._mapHpGaugeWindowY = Number(args.y);
		if (SceneManager._scene && SceneManager._scene._mapHpGaugeWindow) {
			SceneManager._scene._mapHpGaugeWindow.x = Number(args.x);
			SceneManager._scene._mapHpGaugeWindow.y = Number(args.y);
			SceneManager._scene._mapHpGaugeWindow._baseX = Number(args.x);
		}
	});

	//-----------------------------------------------------------------------------
	// Game_System
	//

	Game_System.prototype.isVisibleMapHpGauge = function() {
		if (this._visibleMapHpGauge == null) this._visibleMapHpGauge = startVisible;
		return this._visibleMapHpGauge;
	};
	
	Game_System.prototype.setVisibleMapHpGauge = function(flag) {
		this._visibleMapHpGauge = flag;
	};

	Game_System.prototype.isVisibleMapHpGauges = function(gaugeId) {
		if (this._visibleMapHpGauges == null) {
			this._visibleMapHpGauges = [];
			for (let i = 0; i < gauges.length; i++) {
				this._visibleMapHpGauges[i] = gauges[i].type !== '';
			}
		}
		return this._visibleMapHpGauges[gaugeId];
	};

	Game_System.prototype.setVisibleMapHpGauges = function(gaugeId, flag) {
		if (this._visibleMapHpGauges == null) {
			this._visibleMapHpGauges = [];
			for (let i = 0; i < gauges.length; i++) {
				this._visibleMapHpGauges[i] = gauges[i].type !== '';
			}
		}
		this._visibleMapHpGauges[gaugeId] = flag;
	};

	//-----------------------------------------------------------------------------
	// Window_MapHpGauge
	//

	function Window_MapHpGauge() {
		this.initialize(...arguments);
	}

	Window_MapHpGauge.prototype = Object.create(Window_Base.prototype);
	Window_MapHpGauge.prototype.constructor = Window_MapHpGauge;

	Window_MapHpGauge.prototype.initialize = function() {
		let x, y;
		if (SceneManager.isNextScene(Scene_Battle)) {
			x = gaugeWindowBattleX;
			y = gaugeWindowBattleY;
		} else {
			x = $gameSystem._mapHpGaugeWindowX != null ? $gameSystem._mapHpGaugeWindowX : gaugeWindowX;
			y = $gameSystem._mapHpGaugeWindowY != null ? $gameSystem._mapHpGaugeWindowY : gaugeWindowY;
		}
		const width = gaugeWindowWidth;
		const height = gaugeWindowHeight;
		const rect = new Rectangle(x, y, width, height);
		Window_Base.prototype.initialize.call(this, rect);
		this.openness = $gameSystem.isVisibleMapHpGauge() ? 255 : 0;
		this.opacity = windowOpacity;
		this._gaugeParams = [];
		this._gaugeVisible = [];
		for (let i = 0; i < gauges.length; i++) {
			this._gaugeParams.push({param: -1, max: -1});
			this._gaugeVisible[i] = $gameSystem.isVisibleMapHpGauges(i);
		}
		this._icons = [];
		this._gold = 0;
		this._actorId = -1;
		this._faceName = '';
		this._faceIndex = '';
		this._shakeDuration = 0;
		this._baseX = x;
		this._needFaceRefresh = false;
		this._hideCount = 0;
	};

	Window_MapHpGauge.prototype.lineHeight = function() {
		return this._lineHeight || 36;
	};

	Window_MapHpGauge.prototype.itemPadding = function() {
		return 8;
	};

	Window_MapHpGauge.prototype.setShake = function(power) {
		this._shakeDuration = power;
	};

	Window_MapHpGauge.prototype.update = function() {
		Window_Base.prototype.update.call(this);
		if (this.updateVisibility()) {
			this.open();
			if (this.isNeedRefresh()) {
				const actor = $gameParty.leader();
				for (let i = 0; i < gauges.length; i++) {
					this._gaugeVisible[i] = $gameSystem.isVisibleMapHpGauges(i);
					const gauge = gauges[i];
					if (gauge.type === 'HP') {
						this._gaugeParams[i].param = actor.hp;
						this._gaugeParams[i].max = actor.mhp;
					} else if (gauge.type === 'MP') {
						this._gaugeParams[i].param = actor.mp;
						this._gaugeParams[i].max = actor.mmp;
					} else if (gauge.type === 'TP') {
						this._gaugeParams[i].param = actor.tp;
						this._gaugeParams[i].max = actor.maxTp();
					} else if (gauge.type === 'LV') {
						this._gaugeParams[i].param = actor.currentExp();
						this._gaugeParams[i].max = actor.nextLevelExp();
						this._gaugeParams[i].subParam = actor.level;
					} else if (gauge.type === 'VN') {
						this._gaugeParams[i].param = $gameVariables.value(gauge.param);
						this._gaugeParams[i].max = $gameVariables.value(gauge.max);
					}
				}
				this._icons = actor.stateIcons().concat(actor.buffIcons());
				this._gold = $gameParty.gold();
				this._actorId = actor.actorId();
				this._faceName = actor.faceName();
				this._faceIndex = actor.faceIndex();
				this.refresh();
			}
			this.updateShake();
			this.updateOpacity();
		} else {
			this.close();
		}
	};

	Window_MapHpGauge.prototype.updateVisibility = function() {
		if (!$gameSystem.isVisibleMapHpGauge()) {
			return false;
		}
		if ($gameParty.inBattle()) {
			return true;
		}
		if ((eventBusyHide && $gameMap.isEventRunning()) ||
				(messageBusyHide && $gameMessage.isBusy())) {
			this._hideCount++;
		} else {
			this._hideCount = 0;
		}
		return this._hideCount < 10 && $gameParty.leader();
	};

	Window_MapHpGauge.prototype.isNeedRefresh = function() {
		const actor = $gameParty.leader();
		if (actor) {
			let result = false;
			if (this._actorId !== actor.actorId()) {
				this.setShake(1);
				return true;
			}
			for (let i = 0; i < gauges.length; i++) {
				if (this._gaugeVisible[i] !== $gameSystem.isVisibleMapHpGauges(i)) {
					result = true;
				}
				const gauge = gauges[i];
				const gaugeParam = this._gaugeParams[i];
				if (gauge.type === 'HP') {
					if (gaugeParam.param !== actor.hp || gaugeParam.max !== actor.mhp) {
						if (gaugeParam.param > actor.hp) {
							this.setShake(shakeTime);
						}
						result = true;
					}
				} else if (gauge.type === 'MP') {
					if (gaugeParam.param !== actor.mp || gaugeParam.max !== actor.mmp) {
						result = true;
					}
				} else if (gauge.type === 'TP') {
					if (gaugeParam.param !== actor.tp || gaugeParam.max !== actor.maxTp()) {
						result = true;
					}
				} else if (gauge.type === 'LV') {
					if (gaugeParam.param !== actor.currentExp() ||
							gaugeParam.max !== actor.nextLevelExp() ||
							gaugeParam.subParam !== actor.level) {
						result = true;
					}
				} else if (gauge.type === 'VN') {
					if (gaugeParam.param !== $gameVariables.value(gauge.param) ||
							gaugeParam.max !== $gameVariables.value(gauge.max)) {
						result = true;
					}
				}
			}
			if (stateIconMax > 0) {
				const icons = actor.stateIcons().concat(actor.buffIcons());
				if (this._icons.toString() !== icons.toString()) {
					result = true;
				}
			}
			if (goldWidth > 0 && this._gold !== $gameParty.gold()) {
				result = true;
			}
			if (this._needFaceRefresh) {
				this.refreshFace();
				if (!this._needFaceRefresh) {
					result = true;
				}
			}
			return result;
		}
		return false;
	};

	Window_MapHpGauge.prototype.updateShake = function() {
		if (this._shakeDuration > 0) {
			this._shakeDuration--;
			this.x = this._baseX;
			if (this._shakeDuration > 0) {
				this.x += Math.floor(Math.sin((this._shakeDuration % 10) * Math.PI / 5) * 8);
			}
		}
	};

	Window_MapHpGauge.prototype.updateOpacity = function() {
		if (this.x < $gamePlayer.screenX() + 24 &&
				this.x + gaugeWindowWidth > $gamePlayer.screenX() - 24 &&
				this.y < $gamePlayer.screenY() &&
				this.y + gaugeWindowHeight > $gamePlayer.screenY() - 48) {
			this.opacity = Math.min(collideOpacity, windowOpacity);
			this.contentsOpacity = collideOpacity;
		} else {
			this.opacity = windowOpacity;
			this.contentsOpacity = 255;
		}
	};

	Window_MapHpGauge.prototype.refresh = function() {
		this.contents.clear();
		const actor = $gameParty.leader();
		if (actor) {
			this.refreshFace();
			for (let i = 0; i < gauges.length; i++) {
				if (!$gameSystem.isVisibleMapHpGauges(i)) {
					continue;
				}
				const gauge = gauges[i];
				this._lineHeight = gauge.height;
				this.contents.fontSize = gauge.fontSize;
				if (gauge.type === 'HP') {
					this.drawActorHpGauge(actor, gauge.x, gauge.y, gauge.width);
				} else if (gauge.type === 'MP') {
					this.drawActorMpGauge(actor, gauge.x, gauge.y, gauge.width);
				} else if (gauge.type === 'TP') {
					this.drawActorTpGauge(actor, gauge.x, gauge.y, gauge.width);
				} else if (gauge.type === 'LV') {
					this.drawLvGauge(actor, gauge);
				} else if (gauge.type === 'VN') {
					this.drawVnGauge(this._gaugeParams[i], gauge);
				}
			}
			for (let i = 0; i < stateIconMax; i++) {
				if (!this._icons[i]) break;
				const x = stateIconX + i * Math.floor(ImageManager.iconWidth * stateIconScale / 100);
				this.drawIcon(this._icons[i], x, stateIconY);
			}
			if (goldWidth > 0) {
				this.drawCurrencyValue(this._gold, TextManager.currencyUnit, goldX, goldY, goldWidth);
			}
			this._lineHeight = 36;
			this.resetFontSettings();
		}
	};

	Window_MapHpGauge.prototype.drawActorHpGauge = function(actor, x, y, width) {
		const rate = actor.hp / actor.mhp;
		const color1 = ColorManager.hpGaugeColor1();
		const color2 = ColorManager.hpGaugeColor2();
		this.drawGaugeRect(x, y, width, rate, color1, color2);
		this.changeTextColor(ColorManager.systemColor());
		this.drawText(TextManager.hpA, x, y, 44);
		this.drawCurrentAndMax(actor.hp, actor.mhp, x, y, width, ColorManager.hpColor(actor), ColorManager.normalColor());
	};

	Window_MapHpGauge.prototype.drawActorMpGauge = function(actor, x, y, width) {
		const rate = actor.mmp > 0 ? actor.mp / actor.mmp : 0;
		const color1 = ColorManager.mpGaugeColor1();
		const color2 = ColorManager.mpGaugeColor2();
		this.drawGaugeRect(x, y, width, rate, color1, color2);
		this.changeTextColor(ColorManager.systemColor());
		this.drawText(TextManager.mpA, x, y, 44);
		this.drawCurrentAndMax(actor.mp, actor.mmp, x, y, width, ColorManager.mpColor(actor), ColorManager.normalColor());
	};

	Window_MapHpGauge.prototype.drawActorTpGauge = function(actor, x, y, width) {
		const rate = actor.tp / actor.maxTp();
		const color1 = ColorManager.tpGaugeColor1();
		const color2 = ColorManager.tpGaugeColor2();
		this.drawGaugeRect(x, y, width, rate, color1, color2);
		this.changeTextColor(ColorManager.systemColor());
		this.drawText(TextManager.tpA, x, y, 44);
		this.changeTextColor(ColorManager.tpColor(actor));
		this.drawText(actor.tp, x + width - 64, y, 64, 'right');
	};

	Window_MapHpGauge.prototype.drawCurrentAndMax = function(current, max, x, y, width, color1, color2) {
		const labelWidth = this.textWidth(TextManager.hpA);
		const valueWidth = this.textWidth('0000');
		const slashWidth = this.textWidth('/');
		const x1 = x + width - valueWidth;
		const x2 = x1 - slashWidth;
		const x3 = x2 - valueWidth;
		if (x3 >= x + labelWidth) {
			this.changeTextColor(color1);
			this.drawText(current, x3, y, valueWidth, 'right');
			this.changeTextColor(color2);
			this.drawText('/', x2, y, slashWidth, 'right');
			this.drawText(max, x1, y, valueWidth, 'right');
		} else {
			this.changeTextColor(color1);
			this.drawText(current, x1, y, valueWidth, 'right');
		}
	};

	Window_MapHpGauge.prototype.drawIcon = function(iconIndex, x, y) {
		const bitmap = ImageManager.loadSystem('IconSet');
		const pw = ImageManager.iconWidth;
		const ph = ImageManager.iconHeight;
		const sx = (iconIndex % 16) * pw;
		const sy = Math.floor(iconIndex / 16) * ph;
		const dw = Math.floor(pw * stateIconScale / 100);
		const dh = Math.floor(ph * stateIconScale / 100);
		const lastPaintOpacity = this.contents.paintOpacity;
		this.contents.paintOpacity = stateIconOpacity;
		this.contents.blt(bitmap, sx, sy, pw, ph, x, y, dw, dh);
		this.contents.paintOpacity = lastPaintOpacity;
	};
	
	Window_MapHpGauge.prototype.drawLvGauge = function(actor, gauge) {
		let value1, value2, rate;
		if (actor.isMaxLevel()) {
			value1 = '-------';
			value2 = '-------';
			rate = 1;
		} else {
			const n = actor.currentLevelExp();
			value1 = actor.currentExp() - n;
			value2 = actor.nextLevelExp() - n;
			rate = value1 / value2;
		}
		this.drawGaugeRect(gauge.x, gauge.y, gauge.width, rate, gauge.color[0], gauge.color[1]);
		this.changeTextColor(ColorManager.systemColor());
		this.drawText(TextManager.levelA, gauge.x, gauge.y, 44);
		const color = ColorManager.normalColor();
		this.changeTextColor(color);
		const width = this.textWidth(TextManager.levelA) + 4;
		this.drawText(actor.level, gauge.x + width, gauge.y, 44);
		const remainWidth = gauge.width - width - this.textWidth(String(actor.level));
		this.drawCurrentAndMax(value1, value2, gauge.x + gauge.width - remainWidth, gauge.y, remainWidth, color, color);
	};
	
	Window_MapHpGauge.prototype.drawVnGauge = function(params, gauge) {
		const rate = params.max === 0 ? 0 : params.param / params.max;
		this.drawGaugeRect(gauge.x, gauge.y, gauge.width, rate, gauge.color[0], gauge.color[1]);
		this.changeTextColor(ColorManager.systemColor());
		this.drawText(gauge.name, gauge.x, gauge.y, this.textWidth(gauge.name));
		this.changeTextColor(ColorManager.normalColor());
		if (vnMax) {
			this.drawVnCurrentAndMax(gauge.name, params.param, params.max, gauge.x, gauge.y, gauge.width);
		} else {
			this.drawText(params.param, gauge.x + gauge.width - 64, gauge.y, 64, 'right');
		}
	};
	
	Window_MapHpGauge.prototype.drawVnCurrentAndMax = function(name, current, max, x, y, width) {
		const labelWidth = this.textWidth(name);
		const valueWidth = this.textWidth('0' + max);
		const slashWidth = this.textWidth('/');
		const x1 = x + width - valueWidth;
		const x2 = x1 - slashWidth;
		const x3 = x2 - valueWidth;
		this.changeTextColor(ColorManager.normalColor());
		if (x3 >= x + labelWidth) {
			this.drawText(current, x3, y, valueWidth, 'right');
			this.drawText('/', x2, y, slashWidth, 'right');
			this.drawText(max, x1, y, valueWidth, 'right');
		} else {
			this.drawText(current, x1, y, valueWidth, 'right');
		}
	};

	Window_MapHpGauge.prototype.drawGaugeRect = function(x, y, width, rate, color1, color2) {
		const fillW = Math.floor(width * rate);
		const gaugeY = y + this.lineHeight() - 8;
		const gaugeH = 6;
		this.contents.fillRect(x, gaugeY, width, gaugeH, ColorManager.gaugeBackColor());
		this.contents.gradientFillRect(x, gaugeY, fillW, gaugeH, color1, color2);
	};

	Window_MapHpGauge.prototype.refreshFace = function() {
		if (faceOffsetX === -1000) {
			return;
		}
		const actor = $gameParty.leader();
		const bitmap = ImageManager.loadFace(actor.faceName());
		this._needFaceRefresh = !bitmap.isReady();
		if (!this._needFaceRefresh) {
			const x = gaugeWindowWidth - 144 + faceOffsetX;
			const y = faceOffsetY;
			const height = Math.min(gaugeWindowHeight, 144);
			this.drawFace(actor.faceName(), actor.faceIndex(), x, y, 144, height);
		}
	};

	//-----------------------------------------------------------------------------
	// Scene_Base
	//

	Scene_Base.prototype.createMapHpGaugeWindow = function() {
		this._mapHpGaugeWindow = new Window_MapHpGauge();
		this.addChild(this._mapHpGaugeWindow);
	};

	//-----------------------------------------------------------------------------
	// Scene_Map
	//

	const _Scene_Map_createDisplayObjects = Scene_Map.prototype.createDisplayObjects;
	Scene_Map.prototype.createDisplayObjects = function() {
		_Scene_Map_createDisplayObjects.call(this);
		this.createMapHpGaugeWindow();
	};

	const _Scene_Map_terminate = Scene_Map.prototype.terminate;
	Scene_Map.prototype.terminate = function() {
		if (!SceneManager.isNextScene(Scene_Battle) && !SceneManager.isNextScene(Scene_Map)) {
			this._mapHpGaugeWindow.hide();
		}
		_Scene_Map_terminate.call(this);
	};
	
	const _Scene_Map_stop = Scene_Map.prototype.stop;
	Scene_Map.prototype.stop = function() {
		_Scene_Map_stop.call(this);
		// マップ間の移動時は閉じない
		if (this._mapHpGaugeWindow && !SceneManager.isNextScene(Scene_Map)) {
			this._mapHpGaugeWindow.close();
		}
	};
	
	//-----------------------------------------------------------------------------
	// Scene_Battle
	//

	const _Scene_Battle_createDisplayObjects = Scene_Battle.prototype.createDisplayObjects;
	Scene_Battle.prototype.createDisplayObjects = function() {
		_Scene_Battle_createDisplayObjects.call(this);
		if (useBattleScene) {
			this.createMapHpGaugeWindow();
		}
	};

	const _Scene_Battle_terminate = Scene_Battle.prototype.terminate;
	Scene_Battle.prototype.terminate = function() {
		_Scene_Battle_terminate.call(this);
		if (this._mapHpGaugeWindow) {
			this.removeChild(this._mapHpGaugeWindow);
		}
	};

})();